'use strict';

/* --------------------------------------------------------------
 product_hover.js 2022-01-29
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2022 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Widget that is used for the hover functionality
 * of the product tiles. It includes the functionality
 * for the image gallery inside the tile
 */
gambio.widgets.module('product_hover', [gambio.source + '/libs/events'], function (data) {

    'use strict';

    // ########## VARIABLE INITIALIZATION ##########

    var $this = $(this),
        $window = $(window),
        $body = $('body'),
        $container = null,
        timer = null,
        componentId = null,
        clickTimer = 0,
        defaults = {
        delay: 50, // Delay in ms after which a hovered element gets closed after mouseleave
        flyoverClass: 'flyover', // Class that gets added to every flyover
        scope: '', // Sets the scope selector for the mouseover events
        container: '#wrapper', // Container selector which is the boundary for the cloned element
        productUrlSelector: '.product-url' // a tag selector of product's url
    },
        options = $.extend(true, {}, defaults, data),
        module = {};

    // ########## HELPER FUNCTIONS ##########

    /**
     * Helper function to remove the opened flyovers that
     * were appended to the body by this component
     * @private
     */
    var _removeFlyover = function _removeFlyover(all) {
        var $flyover = $body.children('.' + options.flyoverClass);
        $flyover = all ? $flyover : $flyover.filter('.product-hover-' + componentId);

        $flyover.remove();
    };

    /**
     * Helper function that replaces the preloader
     * images with the real thumbnail images on
     * layer creation. This is needed to save
     * bandwidth
     * @param       {object}    $clone      jQuery selection of the layer
     * @private
     */
    var _loadImages = function _loadImages($clone) {
        $clone.find('.thumbnails img').each(function () {

            var $self = $(this),
                $img = $('<img />'),
                dataset = $self.data(),
                src = dataset.thumbSrc || dataset.src,
                $parentListItem = null;

            $img.on('load', function () {
                $parentListItem = $self.closest('li');
                $parentListItem.addClass('loaded').css({
                    'background': '#FFFFFF url("' + src + '") no-repeat center',
                    'background-size': 'contain'
                }).find('img, .align-helper').remove();
            }).attr('src', src);
        });
    };

    // ########## EVENT HANDLER ##########

    /**
     * Handler for the click event on the thumbnail
     * images. After a click on such an image the
     * main image of the hover element gets replaced
     * with the bigger version of the thumbnail image
     * @param       {object}        e       jQuery event object
     * @private
     */
    var _mouseEnterThumbHandler = function _mouseEnterThumbHandler(e) {
        e.preventDefault();

        var $img = $(this),
            $container = $img.closest('.' + options.flyoverClass),
            dataSrc = $img.css('background-image');

        // Change path to big images and remove quotes
        dataSrc = dataSrc.replace('/thumbnail_images/', '/info_images/').replace('/gallery_images/', '/thumbnail_images/').replace(/["']/gm, '');

        // Remove "url()"
        var matches = dataSrc.match(/url\((.+)\)/);
        if (matches && matches[1]) {
            dataSrc = matches[1];
        }

        if (dataSrc) {
            $container.find('.product-hover-main-image img').attr('src', dataSrc);
        }
    };

    /**
     * Event handler for the mouse leave event of the
     * hovered element. It sets a timer to remove the
     * hover element after a certain time
     * @param       {object}    e       jQuery event object
     * @private
     */
    var _mouseLeaveHandler = function _mouseLeaveHandler(e) {
        e.stopPropagation();
        timer = timer ? clearTimeout(timer) : null;
        timer = window.setTimeout(_removeFlyover, options.delay);
    };

    /**
     * Event handler for the mouse enter event on both
     * elements (initial & hovered element).
     * It clones the initial element and adds the clone
     * to the body. It additionally adds functionality
     * for the image gallery inside the hovered element
     * @param       {object}        e       jQuery event object
     * @private
     */
    var _mouseEnterHandler = function _mouseEnterHandler(e) {
        e.stopPropagation();

        var $self = $(this),
            $clone = null,
            $target = $body,
            uid = $self.data().uid || parseInt(Math.random() * 10000, 10),
            $flyover = $target.children('.' + options.flyoverClass + '.product-hover-' + componentId + '[data-product_hover-uid="' + uid + '"]'),
            offset = $self.offset();

        timer = timer ? clearTimeout(timer) : null;

        // Check if flyover needs to be created
        if (!$self.hasClass(options.flyoverClass) && !$flyover.length) {
            // Remove old opened flyovers
            _removeFlyover(true);
            $this.trigger(jse.libs.theme.events.OPEN_FLYOUT(), $this);

            // Add a UID for identification to th hovered object
            $self.attr('data-product_hover-uid', uid).data('uid', uid);

            // Generate the markup
            $clone = $self.clone(true);

            // Replace the preloader images with the thumbnail images
            _loadImages($clone);

            // Set the positioning of the layer
            $clone.addClass(options.flyoverClass + ' product-hover-' + componentId).css({
                'position': 'absolute',
                'left': offset.left,
                'top': offset.top,
                'width': $self[0].getBoundingClientRect().width,
                'height': $self[0].getBoundingClientRect().height
            });

            // Add event listener to the hover elements
            $clone.on('mouseenter', _mouseEnterHandler).on('mouseleave', _mouseLeaveHandler).on('mouseenter', '.thumbnails', _mouseEnterThumbHandler).on('click', _clickHandler);

            // Add the element to the body element
            $body.append($clone);

            if ($container.offset().left > $clone.offset().left) {
                $clone.addClass('gallery-right');
            }
        }
    };

    /**
     * Handler for the window resize event. It
     * recalculates the position of the overlays
     * @private
     */
    var _resizeHandler = function _resizeHandler() {

        var $flyover = $body.children('.' + options.flyoverClass + '.product-hover-' + componentId);

        $flyover.each(function () {
            var $self = $(this),
                uid = $self.data().uid,
                $source = $this.find('[data-product_hover-uid="' + uid + '"]'),
                offset = $source.offset();

            $self.css({
                left: offset.left,
                top: offset.top,
                width: 2 * $source.outerWidth()
            });
        });
    };

    /**
     * Event handler that closes the flyovers
     * if another flyover opens on the page
     * @param       {object}        e           jQuery event object
     * @param       {object}        d           jQuery selection of the event emitter
     * @private
     */
    var _closeLayers = function _closeLayers(e, d) {
        if ($this !== d) {
            _removeFlyover();
        }
    };

    /**
     * Event handler that makes the flyover and product image clickable linking to the product details page
     *
     * @param       {object}        e           jQuery event object
     * @private
     */
    var _clickHandler = function _clickHandler(e) {
        var $container = $(this);

        if ($(this).hasClass('product-container') === false) {
            $container = $(this).closest('.product-container');
        }

        var $link = $container.find(options.productUrlSelector).first();

        function callback() {
            if ($link.length) {
                var url = $link.attr('href');

                if (url !== undefined) {
                    e.stopPropagation();
                    e.preventDefault();

                    // prevent double _clickHandler actions
                    if (new Date().getTime() - clickTimer < 100) {
                        return;
                    } else {
                        clickTimer = new Date().getTime();
                    }

                    switch (e.which) {
                        // left click
                        case 1:
                            if (e.ctrlKey) {
                                window.open(url, '_blank');
                                return;
                            }
                            break;

                        // middle click
                        case 2:
                            window.open(url, '_blank');
                            return;
                            break;

                        // right click
                        case 3:
                            return;
                    }

                    location.href = url;
                }
            }
        }

        jse.libs.hooks.execute(jse.libs.hooks.keys.shop.product.listing.hover, { container: $container }, 500).then(callback).catch(callback);
    };

    // ########## INITIALIZATION ##########

    /**
     * Init function of the widget
     * @constructor
     */
    module.init = function (done) {

        componentId = parseInt(Math.random() * 10000, 10);
        $container = $(options.container);

        $this.on('touchstart', function () {
            // Workaround for tablet navigation problem
            $this.off('mouseenter mouseleave');
        }).on('touchend', function () {
            $this.off('mouseenter', options.scope + ' .product-container', _mouseEnterHandler).off('mouseleave', options.scope + ' .product-container', _mouseLeaveHandler);
        }).on('mouseenter', options.scope + ' .product-container', _mouseEnterHandler).on('mouseleave', options.scope + ' .product-container', _mouseLeaveHandler);

        $this.find('.product-container .product-image').on('click mouseup', _clickHandler);

        $body.on(jse.libs.theme.events.OPEN_FLYOUT(), _closeLayers);

        $window.on('resize', _resizeHandler);

        done();
    };

    // Return data to widget engine
    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
